/**
 * URL Processing
 *
 * Copyright (C) 2000 Jeffrey Fulmer <jdfulmer@armstrong.com>
 * This file is part of Siege
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */
#include <setup.h>
#include <joedog/joedog.h>
#define PLENGTH 84

/** 
 * ALERT: using hardcoded array lengths below,
 * if you change this array, then redefine PLENGTH
 * 
 * Currently http(prot[25]) and https(prot[26]) are 
 * the only supported protocols.  But all w3c supported 
 * protocols are listed for URL evaluation.
 */  
static char *prot[] = {
  "about:",      "addrbook:",  "acap:",      "afp:",
  "afs:",        "callto:",    "chttp:",     "cid:",
  "clsid:",      "data:",      "date:",      "DAV:",
  "dns:",        "eid:",       "fax:",       "file:",
  "finger:",     "freenet:",   "ftp:",       "gopher:",
  "gsm:",        "h323:",      "h324:",      "hdl:",
  "hnews:",      "http:",      "https:",     "iioploc:",
  "ilu:",        "imap:",      "IOR:",       "irc:",
  "isbn:",       "java:",      "JavaRMI:",   "javascript:",
  "jdbc:",       "ldap:",      "lid:",       "lifn:",
  "livescript:", "lrq:",       "mailto:",    "mailserver:",
  "md5:",        "mid:",       "mocha:",     "modem:",
  "news:",       "nfs:",       "nntp:",      "opaquelocktoken:"
  "path:",       "phone:",     "pop:",       "pop3:",
  "printer:",    "prospero:",  "res:",       "rtsp:",
  "rvp:",        "rwhois:",    "rx:",        "sdp:",
  "sip:",        "shttp:",     "snews:",     "STANF:",
  "t120:",       "tel:",       "telephone:", "telnet:",
  "tip:",        "tn3270:",    "tv:",        "uuid:",
  "urn:",        "vemmi:",     "videotex:",  "view:",
  "wais:",       "whois++:",   "whodp:",     "z39.50r:",
  "z39.50s:"
  
};

/**
 * int value of the length of the protocol 
 * string passed to the function.
 */     
int
protocol_length( char *url )
{
  int x;
  /** 
   * hardcoded protocol length!! 
   * see explanation above...
   */
  for( x = 0; x < PLENGTH; x ++ ){ 
    if( strncasecmp( url, prot[x], strlen( prot[x] )) == 0 )
      return strlen( prot[x] );
  } 
  return 0;	
}

/**
 * boolean, returns true if the protocol is 
 * supported by siege, false if it is not.
 */ 
int
is_supported( char* url )
{
  if( strncasecmp( url, prot[25], strlen( prot[25] )) == 0 )
    return TRUE;
  if( strncasecmp( url, prot[26], strlen( prot[26] )) == 0 )
    #ifdef HAVE_SSL
      return TRUE;
    #else
      return FALSE;
    #endif /* HAVE_SSL */
  else
    return FALSE;
}

/**
 * Returns a char pointer of the default
 * port.  Why not just return an int? Legacy.
 */
int
get_default_port( char *p )
{
  /* match the prot array, could have 
     chosen strlen -1 as well.       */
  strcat( p, ":" );

  if( strncasecmp( p, prot[25], strlen( prot[25] )) == 0 )
    return 80;
  if( strncasecmp( p, prot[26], strlen( prot[26] )) == 0 )
    #ifdef HAVE_SSL
      return 443;
    #else 
      return 80;     /* this might get me in trouble! */
    #endif /* HAVE_SSL */
  else
    return 80;
}

/**
 * insert_childid 
 * replaces all '+' characters in POST 
 * data with numbers derived from the process id.
 * return void
 */
void
insert_childid( URL *U, int mypid )
{
  int    i,j;
  char   *c, *f, *l;
  char   pidbuf[6];

  sprintf( pidbuf, "%05.5d", mypid );

  for( i=0; i<U->index; i++ ){
    if( U->posttemp[i] ){
      f = strchr(U->posttemp[i], '+');
      l = strrchr(U->posttemp[i], '+');
      /* Start at last occurrence of '+' and move to first */
      for (j=sizeof(pidbuf)-1, c=l; c >= f; j--, c--){
        if (*c == '+'){
          *c = pidbuf[j-1];
        }
        if (j == 0){
          j=sizeof(pidbuf)-1;   /* Start over */
        }
      }
      /* Now that we're done, copy the new template to data */
      strcpy(U->postdata[i], U->posttemp[i]);
    }
  }
}

/**
 * build_from_template 
 * builds POST data replacing all '*' 
 * characters with random numbers
 * returns void
 */
void
build_from_template( URL *U, int index, int rand )
{
  char   *s=U->posttemp[index];
  char   *t=U->postdata[index];
  int    i,j,f,l;
  char   buf[9];

  sprintf(buf, "%08.8d", rand);

  f = strchr(s, '*')  -s;
  l = strrchr(s, '*') -s;
  /* Start at last occurrence of '+' and move to first */
  for (j=sizeof(buf)-1, i=l; i >= f; j--, i--){
    if (s[i] == '*'){
      t[i] = buf[j-1];
    } else {
      t[i] = s[i];
    }
    if (j == 0){
      j=sizeof(buf)-1;   /* Start over */
    }
  }
}

/**
 * process_post_data   
 * populates URL->postdata with POST information
 * returns int
 */
int
process_post_data( URL *U, char *datap )
{
  for( ; isspace(*datap); datap++ ){ 
    /* Advance past white space */ 
  }
  if( *datap == '<' ){
    /* Get Data from file */
  } 
  else{
    U->postlen[U->index]  = strlen(datap);
    U->postdata[U->index] = malloc(U->postlen[U->index]+1);
    if (strchr(datap, '+') || strchr(datap, '*')){
      U->posttemp[U->index] = malloc(U->postlen[U->index]+1);
    } 
    else{
      U->posttemp[U->index] = NULL;
    }
    if (!U->postdata[U->index]){
      return 0;                 /* Failure */
    }
    memcpy(U->postdata[U->index], datap, U->postlen[U->index]);
    U->postdata[U->index][U->postlen[U->index]] = 0;
    if( U->posttemp[U->index] ){
      memcpy(U->posttemp[U->index], datap, U->postlen[U->index]);
      U->posttemp[U->index][U->postlen[U->index]] = 0;
    }
  }
}


/**
 * parse_url    populates struct URL with URL 
 * information, the struct is comprised of a
 * series of **pointers which are referenced
 * by u->variable[i].  
 */    
int 
parse_url( URL *U, char *url )
{
  int one, two, thr, fou;  /* placement counters. */
  char *post_cmd=NULL;   

  if( !U ){    
    /* not enough memory! */
    perror( "structure not initialized\n" );
    /* handle in the calling function */
    return 0;
  }

  /* Break the URL component away from POST directives */
  U->postlen  = realloc(U->postlen, sizeof(size_t)*(U->index+1));
  U->postdata = realloc(U->postdata, sizeof(char*)*(U->index+1));
  U->posttemp = realloc(U->posttemp, sizeof(char*)*(U->index+1));
  U->calltype = (void*)realloc(U->calltype, sizeof(url_access_t) * (U->index+1));
  post_cmd = strstr(url, " POST");
  if (post_cmd != NULL) {
    /* How do we deal with handling the multi-headed url_t arrays */
    U->calltype = (void*)realloc(U->calltype, sizeof(url_access_t) * (U->index+1));
    U->calltype[U->index] = URL_POST;
    *post_cmd = 0;
    post_cmd += 5;
    process_post_data( U, post_cmd );
  } 
  else{
    U->calltype[U->index]   = URL_GET;
    U->postdata[U->index]   = NULL;
    U->posttemp[U->index]   = NULL;
    U->postlen[U->index]    = 0;
  }

  if(( one = protocol_length( url )) > 0 && is_supported( url ) == TRUE )
    one += 2;
  else if(( one = protocol_length( url )) > 0 && is_supported( url ) == FALSE )
    joe_fatal( "unsupported protocol" );
  else
    one = 0;  /* no specified protocol, assuming http: */

  two = one;
  while( url[two] && url[two] != ':' && url[two] != '/' ) {
    two++;
  }
  if( url[two] == ':' ){
    fou = two;
    while( url[two] && url[two] != '/' ){
      two++;
    }
  }
  else{
    fou = two;
  }
  if( url[two] == '/' )
    thr = two;
  else
    thr = strlen( url );
  U->protocol = (void*)realloc(U->protocol, sizeof(char *) * ( U->index +1 ));
  if( one == 0 )
    U->protocol[U->index] = (char *)strdup( "http://" );
  else
    U->protocol[U->index] = (char *)strdup(substring( url, 0, one-3 ));
  U->hostname = (void*)realloc( U->hostname, sizeof(char *) * ( U->index +1 ));
  U->hostname[U->index] = (char *)strdup(substring( url, one, ( fou - one )));
  if( fou == two )
    U->port = get_default_port( U->protocol[U->index] );
  else
    U->port = atoi(substring( url, fou+1, (thr-(fou+1))));
  U->pathname = (void*)realloc( U->pathname, sizeof(char *) * ( U->index +1 ));
  if(( U->pathname[U->index] = (char *)strdup(substring( url, thr, strlen( url )))) == NULL )
    U->pathname[U->index] = (char *)strdup( "/" );

  U->index ++;
  free( url );
  return 1;
} 

