/*
 * OpenGatekeeper
 * 
 * Copyright (c) Egoboo Ltd. 1999-2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: opengate.cxx,v $
 * Revision 1.19  2000/05/12 13:58:24  aunitt
 * Renamed RASLog module to OpengateLog
 *
 * Revision 1.18  2000/05/05 11:11:02  aunitt
 * CallTable now needs environ (due to bandwidth management)
 *
 * Revision 1.17  2000/05/02 16:30:11  aunitt
 * Moved version information to Version.h
 *
 * Revision 1.16  2000/05/02 10:20:47  aunitt
 * Removed local address changes as they broke other things.
 *
 * Revision 1.15  2000/04/29 12:53:10  aunitt
 * Improved working in a multi homed environment.
 *
 * Revision 1.14  2000/04/27 17:42:12  aunitt
 * Removed ifdefs as Windows multicast is now working.
 *
 * Revision 1.13  2000/04/26 17:06:08  aunitt
 * Added initial support for registration time to live features.
 *
 * Revision 1.12  2000/04/21 13:38:50  aunitt
 * Updated to release 6
 *
 * Revision 1.11  2000/04/13 18:43:34  aunitt
 * Moved open for RASLog to the constructor.
 *
 * Revision 1.10  2000/04/10 19:09:01  aunitt
 * Changes to support gatekeeper routed calls.
 *
 * Revision 1.9  2000/04/05 15:13:07  aunitt
 * Added RASLog object. Thanks to Tony Genovese.
 *
 * Revision 1.8  2000/03/23 00:01:29  aunitt
 * Changed name of configuration section from Opengate to System.
 *
 * Revision 1.7  2000/03/21 21:10:56  aunitt
 * Fixed bug in not handling OnStop properly. We were not supposed to return until
 * we had stopped.
 * Fixed bug in multicast for Win32, we still need to make sure that we are
 * members of the multicast group.
 *
 * Revision 1.6  2000/03/12 21:17:12  aunitt
 * Fixed bug in not handling  messages from endpoints that were not registered
 *
 * Revision 1.5  2000/02/27 20:49:42  aunitt
 * Added windows support. Fixed replies to the wrong RAS address
 *
 * Revision 1.4  2000/02/15 20:44:13  aunitt
 * Added support for using system log for debugging. Log level is set in the configuration file
 *
 * Revision 1.3  2000/02/09 22:29:52  aunitt
 * Added support for reading settings from the configuration file
 *
 * Revision 1.2  2000/02/06 16:52:11  aunitt
 * Removed command line parsing because it doesn't make sense any more as we are a service process. Later on we should add parsing of a configuration file
 *
 * Revision 1.1.1.1  2000/02/01 22:25:34  aunitt
 * Initial revision
 *
 *
 */

#if !defined(WIN32)
#include <signal.h>
#endif

#include <ptlib.h>
#include <ptlib/sockets.h>
#include <q931.h>               // To stop "ostream" ambiguous symbol error
#include "RasThread.h"
#include "CallThread.h"
#include "Environ.h"
#include "MulticastThread.h"
#include "Version.h"
#include "opengate.h"

static PSemaphore TerminationSema(0,1); // Main will wait on this semaphore for termination
static PSemaphore TerminatedSema(0,1);  // OnStop then waits for this until finishing....

PCREATE_PROCESS(opengate)

opengate::opengate()
	: PServiceProcess("Egoboo", "opengate",
	                  GKVER_MAJOR, GKVER_MINOR, GKVER_STATUS, GKVER_BUILD
	                 )
{
}

BOOL opengate::OnStart()
{
	// Nothing to do...
	return TRUE;
}

void opengate::OnStop()
{
	// Signal the termination semaphore and Main will die....
	TerminationSema.Signal();
	
	// Now wait for Main to die...
	TerminatedSema.Wait();
}

void opengate::Terminate()
{
	// Hmm, why is Terminate different from OnStop?
	OnStop();
}

#if !defined(WIN32)
// It looks like I need my own signal handler in Unix to stop the PProcess async handler
// killing me before I want to die
void SigHandler( int sig )
{
	switch( sig )
	{
		case SIGINT  :
		case SIGTERM : TerminationSema.Signal();
		               break;
		default      : break;
	}
}
#endif

void opengate::OnControl()
{
	// Nothing to do....
}

void opengate::Main()
{
    Environ MyEnviron;

#if !defined(WIN32)
	signal( SIGINT, SigHandler );
	signal( SIGTERM, SigHandler );
#endif
	// Try and read settings from the configuration file, if they are not
	// there using sensible default

	PConfig Cfg("System");	

	// Set the local address to bind to, this should be useful in multi-homed environments
	PString AkaLocalAddr = Cfg.GetString( "Local Address", "*" );
	MyEnviron.LocalAddr = AkaLocalAddr;
	
    // Set the gatekeeper id, the default is "Opengate: " + hostname
	PString AkaMyId = Cfg.GetString( "Gatekeeper Id", (PString) "Opengate: " + PIPSocket::GetHostName() );
	MyEnviron.MyId = AkaMyId;
//    if ( MyEnviron.LocalAddr == INADDR_ANY )
//        PIPSocket::GetHostAddress( MyEnviron.LocalAddr );
	
	// Set the log level, default is errors only (1) for normal builds and info(3) for
	// debug builds
	Current().SetLogLevel( static_cast<PSystemLog::Level>( Cfg.GetInteger( "Log Level",
#if defined ( _DEBUG )
                                                                           PSystemLog::Info
#else
                                                                           PSystemLog::Error
#endif
                                                                         )
	                                                     )
	                     );
	
	// Create the endpoint table
	MyEnviron.EPTable = new EndpointTable( MyEnviron );

	// Create the call table
    MyEnviron.CTable = new CallTable( MyEnviron );
	
	// Create the RAS log
	MyEnviron.Log = new OpengateLog;
	
	// Create the call signalling thread
	CallThread * CThread = new CallThread( MyEnviron );
	MyEnviron.MyCallAddr = CThread->GetMyCallSignallingAddr();
	
	// Create the RAS thread
	RasThread * RThread = new RasThread( MyEnviron );

	// Create the Multicast receiver thread
	MulticastThread * MCThread = new MulticastThread( MyEnviron );
	
	// Wait for death....
	TerminationSema.Wait();
	
    // Clean up
   	MCThread->Close();
   	RThread->Close();
   	CThread->Close();
   	MCThread->WaitForTermination();
    RThread->WaitForTermination();
    CThread->WaitForTermination();

    delete MCThread;
   	delete RThread;
   	delete CThread;
   	delete MyEnviron.Log;
   	delete MyEnviron.CTable;
   	delete MyEnviron.EPTable;
   	
   	// Tell OnStop we have finished...
	TerminatedSema.Signal();
	
	// Wait for death
	Sleep(10000);
}
