/*
 * main.cxx
 *
 * Auxiliary commands for OpenAM
 *
 * Copyright (c) 1993-2001 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Portable Windows Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): ______________________________________.
 *
 * $Log: cmds.cxx,v $
 * Revision 1.2  2001/09/25 01:10:30  robertj
 * Fixed compile problerm. Still incomplete code.
 *
 * Revision 1.1  2001/09/24 22:39:42  craigs
 * Added commands to play and record data files, esp G.723.1
 *
 */

#include <ptlib.h>
#include <ixjlid.h>

#include "version.h"
#include "main.h"

#define new PNEW

#if HAS_IXJ

#define	G7231_BUFFER_SIZE	24
#define	PCM_BUFFER_SIZE		480

extern PString G7231Ext;
extern PString WAVExt;

static PFile * DetermineType(PArgList & args, const PFilePath & fn, BOOL & isPCM)
{
  BOOL knowType = FALSE;

  // determine file type from filename extension
  if (fn.GetType() *= WAVExt) {
    isPCM    = TRUE;
    knowType = TRUE;
  } else if (fn.GetType() *= G7231Ext) {
    isPCM    = FALSE;
    knowType = TRUE;
  }

  // if file extension not recognised, look for options
  if (!knowType) {
    if (args.HasOption("pcm")) {
      isPCM    = TRUE;
      knowType = TRUE;
    } else if (args.HasOption("g7231")) {
      isPCM    = FALSE;
      knowType = TRUE;
    }
  }

  // error if we could not dtermine the file type
  if (!knowType) {
    PError << "usage: record command requires either --g7231 or --pcm,"
           << "       or filename with " << WAVExt << " or " << G7231Ext << " extension" << endl;
    return NULL;
  }

  // get the correct file type
  if (isPCM) 
    return new PWAVFile;

  return new PFile;
}

static BOOL OpenDevice(OpalIxJDevice & xJack, const PString & ixjDevice)
{
  if (!xJack.Open(ixjDevice)) {
    PError << "error: cannot open device \"" << ixjDevice << "\"" << endl;
    return FALSE;
  }

  xJack.SetLineToLineDirect(0, 1, FALSE);
  xJack.EnableAudio(0, TRUE);
  return TRUE;
}

void OpenAm::RecordFile(PArgList & args)
{
  if (args.GetCount() < 2) {
    PError << "usage: openam record [--pcm|--g7231] -qn fn" << endl;
    return;
  }

  PFilePath fn  = args[1];
  BOOL isPCM = FALSE;

  // create the correct type of file
  PFile * recordFile = DetermineType(args, fn, isPCM);
  if (recordFile == NULL)
    return;

  // open the file
  if (!recordFile->Open(fn, PFile::WriteOnly)) {
    PError << "error: cannot open file \"" << fn << "\"" << endl;
    return;
  }

  // open the device
  if (!args.HasOption('q')) {
    PError << "error: record command requires -q option for Quicknet device" << endl;
    return;
  }

  OpalIxJDevice xJack;
  if (!OpenDevice(xJack, args.GetOptionString('q'))) 
    return;

  // ring handset
  xJack.RingLine(OpalIxJDevice::POTSLine, 0x33);

  // wait for answer
  cout << "Waiting for phone to go offhook...";
  cout.flush();

  while (!xJack.IsLineOffHook(OpalIxJDevice::POTSLine))
    Sleep(100);
  cout << "recording" << endl;

  // start codecs
  if (!xJack.SetReadCodec (OpalIxJDevice::POTSLine,
                           isPCM ? RTP_DataFrame::L16_Mono : RTP_DataFrame::G7231)) {
    PError << "error: error during SetReadCodec" << endl;
    return;
  }
  if (!xJack.SetWriteCodec(OpalIxJDevice::POTSLine,
                           isPCM ? RTP_DataFrame::L16_Mono : RTP_DataFrame::G7231)) {
    PError << "error: error during SetWriteCodec" << endl;
    return;
  }

  // determine the read buffer size
  PINDEX bufferSize;
  if (isPCM)
    bufferSize = xJack.GetReadFrameSize(OpalIxJDevice::POTSLine);
  else
    bufferSize = G7231_BUFFER_SIZE;

  // allocate a buffer
  PBYTEArray buffer;
  buffer.SetSize(bufferSize);

  // start recording until hangup
  while (xJack.IsLineOffHook(OpalIxJDevice::POTSLine)) {
    PINDEX count;
    if (!xJack.ReadFrame(OpalIxJDevice::POTSLine, buffer.GetPointer(), count)) {
      PError << "error: error during ReadFrame" << endl;
      return;
    }
    recordFile->Write(buffer, count);
  }

  // stop recording;
  xJack.StopReadCodec (OpalIxJDevice::POTSLine);
  xJack.StopWriteCodec(OpalIxJDevice::POTSLine);

  // close the file
  recordFile->Close();
  delete recordFile;
}


void OpenAm::PlayFile(PArgList & args)
{
  if (args.GetCount() < 2) {
    PError << "usage: openam play [--pcm|--g7231] -qn fn" << endl;
    return;
  }

  PFilePath fn  = args[1];
  BOOL isPCM = FALSE;

  // create the correct type of file
  PFile * playFile = DetermineType(args, fn, isPCM);
  if (playFile == NULL)
    return;

  // open the file
  if (!playFile->Open(fn, PFile::ReadOnly)) {
    PError << "error: cannot open file \"" << fn << "\"" << endl;
    return;
  }

  // open the device
  if (!args.HasOption('q')) {
    PError << "error: play command requires -q option for Quicknet device" << endl;
    return;
  }

  OpalIxJDevice xJack;
  if (!OpenDevice(xJack, args.GetOptionString('q'))) 
    return;

  // ring handset
  xJack.RingLine(OpalIxJDevice::POTSLine, 0x33);

  // wait for answer
  cout << "Waiting for phone to go offhook...";
  cout.flush();

  while (!xJack.IsLineOffHook(OpalIxJDevice::POTSLine))
    Sleep(100);
  cout << "ok" << endl
       << "Playing " << (isPCM ? "PCM" : "G.723.1") << " message" << endl;


  // start codecs
  if (!xJack.SetReadCodec (OpalIxJDevice::POTSLine,
                           isPCM ? RTP_DataFrame::L16_Mono : RTP_DataFrame::G7231)) {
    PError << "error: error during SetReadCodec" << endl;
    return;
  }
  if (!xJack.SetWriteCodec(OpalIxJDevice::POTSLine,
                           isPCM ? RTP_DataFrame::L16_Mono : RTP_DataFrame::G7231)) {
    PError << "error: error during SetWriteCodec" << endl;
    return;
  }

  // determine the write buffer size
  PINDEX bufferSize;
  if (isPCM)
    bufferSize = xJack.GetReadFrameSize(OpalIxJDevice::POTSLine);
  else
    bufferSize = G7231_BUFFER_SIZE;

  // allocate a buffer
  PBYTEArray buffer;
  buffer.SetSize(bufferSize);

  // start playing until hangup
  while (xJack.IsLineOffHook(OpalIxJDevice::POTSLine)) {
    PINDEX count = 0;
    if (isPCM) {
      if (!playFile->Read(buffer.GetPointer(), bufferSize))
        break;
    } else {
      if (!playFile->Read(buffer.GetPointer(), 1)) 
        break;
      count++;
      static const int frameLen[] = { 24, 20, 4, 1 };
      if (!playFile->Read(buffer.GetPointer()+1, frameLen[buffer[0]&3]))
        break;
    }
    count += playFile->GetLastReadCount();

    PINDEX written;
    if (!xJack.WriteFrame(OpalIxJDevice::POTSLine, buffer.GetPointer(), count, written)) {
      PError << "error: error during WriteFrame" << endl;
      return;
    }
  }

  // stop recording;
  xJack.StopReadCodec (OpalIxJDevice::POTSLine);
  xJack.StopWriteCodec(OpalIxJDevice::POTSLine);

  // close the file
  playFile->Close();
  delete playFile;
}


#endif
